<?php
/**
 * Database Class - Suporta MySQL e SQLite
 */
class Database {
    private static $instance = null;
    private $pdo;
    
    private function __construct() {
        // Carrega config se existir
        $configFile = dirname(__DIR__) . '/config.php';
        
        if (file_exists($configFile)) {
            require_once $configFile;
        }
        
        // MySQL
        if (defined('DB_TYPE') && DB_TYPE === 'mysql') {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ]);
        } 
        // SQLite (fallback)
        else {
            $dbPath = dirname(__DIR__) . '/database/rifa.db';
            $dir = dirname($dbPath);
            if (!is_dir($dir)) mkdir($dir, 0755, true);
            
            $this->pdo = new PDO('sqlite:' . $dbPath);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            $this->pdo->exec('PRAGMA foreign_keys = ON');
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function init() {
        // Verifica se é MySQL ou SQLite
        $isMysql = defined('DB_TYPE') && DB_TYPE === 'mysql';
        
        if ($isMysql) {
            // MySQL já foi inicializado pelo instalador
            return true;
        }
        
        // SQLite - usa schema.sql
        $schemaFile = dirname(__DIR__) . '/database/schema.sql';
        if (file_exists($schemaFile)) {
            $schema = file_get_contents($schemaFile);
            $this->pdo->exec($schema);
        }
        return true;
    }
    
    public function query($sql, $params = []) {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    }
    
    public function fetchOne($sql, $params = []) {
        return $this->query($sql, $params)->fetch();
    }
    
    public function fetchAll($sql, $params = []) {
        return $this->query($sql, $params)->fetchAll();
    }
    
    public function insert($table, $data) {
        $cols = implode(', ', array_keys($data));
        $vals = implode(', ', array_fill(0, count($data), '?'));
        $this->query("INSERT INTO {$table} ({$cols}) VALUES ({$vals})", array_values($data));
        return $this->pdo->lastInsertId();
    }
    
    public function update($table, $data, $where, $params = []) {
        $set = implode(' = ?, ', array_keys($data)) . ' = ?';
        return $this->query("UPDATE {$table} SET {$set} WHERE {$where}", array_merge(array_values($data), $params))->rowCount();
    }
    
    public function delete($table, $where, $params = []) {
        return $this->query("DELETE FROM {$table} WHERE {$where}", $params)->rowCount();
    }
    
    public function exists($table, $where, $params = []) {
        return (bool)$this->fetchOne("SELECT 1 FROM {$table} WHERE {$where} LIMIT 1", $params);
    }
    
    public function lastId() {
        return $this->pdo->lastInsertId();
    }
    
    public function getPdo() {
        return $this->pdo;
    }
}
