<?php
/**
 * ============================================
 * 🛡️ PROTEÇÃO ANTI-CLONE v2.0
 * ============================================
 * 
 * Coloque este arquivo no seu CHECKOUT/SITE REAL
 * Inclua no topo de cada página que quer proteger
 * 
 * Proteções:
 * - Verifica se veio do Cloaker (sessão válida)
 * - Bloqueia iframes (anti-clone)
 * - Bloqueia DevTools
 * - Bloqueia bots e crawlers
 * - Bloqueia acesso direto
 * - Ofusca código fonte
 * - Detecta comportamento suspeito
 */

class AntiClone {
    
    private $config = [
        // Token secreto (mesmo do Cloaker)
        'secret_token' => 'RIFA_PROTECT_2025',
        
        // Tempo máximo da sessão (segundos)
        'session_timeout' => 3600, // 1 hora
        
        // Página para redirecionar se bloqueado
        'block_redirect' => 'https://google.com/',
        
        // Permitir acesso direto (sem vir do cloaker)?
        'allow_direct' => false,
        
        // Bloquear iframes?
        'block_iframe' => true,
        
        // Bloquear DevTools?
        'block_devtools' => true,
        
        // Bloquear bots?
        'block_bots' => true,
        
        // Bloquear países (deixe vazio para permitir todos)
        'blocked_countries' => [],
        
        // IPs na whitelist (sempre permitidos)
        'whitelist_ips' => [],
        
        // Domínios permitidos como referer
        'allowed_referers' => [],
        
        // Ativar logs?
        'enable_logs' => true,
        'log_file' => __DIR__ . '/logs/anticlone.log',
    ];
    
    private $checks = [];
    private $score = 100;
    
    public function __construct($customConfig = []) {
        $this->config = array_merge($this->config, $customConfig);
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }
    
    /**
     * Executa todas as verificações
     */
    public function protect() {
        $ip = $this->getIP();
        
        // Whitelist
        if (in_array($ip, $this->config['whitelist_ips'])) {
            return true;
        }
        
        // Verificações
        $this->checkSession();
        $this->checkBot();
        $this->checkReferer();
        $this->checkHeaders();
        $this->checkCountry($ip);
        
        // Adiciona headers de proteção
        $this->addSecurityHeaders();
        
        // Score muito baixo = bloqueia
        if ($this->score < 50) {
            $this->log($ip, 'BLOCKED', $this->checks);
            $this->block();
        }
        
        // Injeta JavaScript de proteção
        $this->injectProtection();
        
        $this->log($ip, 'ALLOWED', $this->checks);
        return true;
    }
    
    /**
     * Verifica sessão válida do Cloaker
     */
    private function checkSession() {
        // Verifica se tem token na URL (vindo do cloaker)
        if (isset($_GET['_t'])) {
            $token = $_GET['_t'];
            $expected = md5($this->config['secret_token'] . date('Y-m-d'));
            
            if ($token === $expected) {
                $_SESSION['anticlone_valid'] = true;
                $_SESSION['anticlone_time'] = time();
                $_SESSION['anticlone_ip'] = $this->getIP();
                
                // Remove token da URL (limpa)
                $cleanUrl = strtok($_SERVER['REQUEST_URI'], '?');
                $params = $_GET;
                unset($params['_t']);
                if (!empty($params)) {
                    $cleanUrl .= '?' . http_build_query($params);
                }
                header("Location: $cleanUrl");
                exit;
            }
        }
        
        // Verifica sessão existente
        if (!isset($_SESSION['anticlone_valid']) || $_SESSION['anticlone_valid'] !== true) {
            if (!$this->config['allow_direct']) {
                $this->score -= 40;
                $this->checks[] = 'no_session';
            }
            return;
        }
        
        // Verifica timeout
        if (time() - $_SESSION['anticlone_time'] > $this->config['session_timeout']) {
            $this->score -= 30;
            $this->checks[] = 'session_expired';
            return;
        }
        
        // Verifica se IP mudou
        if ($_SESSION['anticlone_ip'] !== $this->getIP()) {
            $this->score -= 20;
            $this->checks[] = 'ip_changed';
        }
    }
    
    /**
     * Detecta bots e crawlers
     */
    private function checkBot() {
        if (!$this->config['block_bots']) return;
        
        $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        $bots = [
            'bot', 'crawler', 'spider', 'scraper', 'curl', 'wget', 'python',
            'java', 'perl', 'ruby', 'go-http', 'httpclient', 'httpunit',
            'phantom', 'headless', 'selenium', 'puppeteer', 'playwright',
            'facebook', 'facebookexternalhit', 'google', 'googlebot',
            'bing', 'bingbot', 'yahoo', 'baidu', 'yandex', 'duckduck',
            'semrush', 'ahrefs', 'moz', 'majestic', 'screaming',
            'httrack', 'offline', 'mirror', 'wget', 'curl',
            'libwww', 'lwp', 'httpie', 'postman', 'insomnia'
        ];
        
        foreach ($bots as $bot) {
            if (strpos($ua, $bot) !== false) {
                $this->score -= 80;
                $this->checks[] = 'bot:' . $bot;
                return;
            }
        }
        
        // User-Agent vazio ou muito curto
        if (empty($ua) || strlen($ua) < 20) {
            $this->score -= 30;
            $this->checks[] = 'suspicious_ua';
        }
    }
    
    /**
     * Verifica referer
     */
    private function checkReferer() {
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        
        // Sem referer (acesso direto)
        if (empty($referer)) {
            if (!$this->config['allow_direct']) {
                $this->score -= 15;
                $this->checks[] = 'no_referer';
            }
            return;
        }
        
        // Verifica se referer está na lista permitida
        if (!empty($this->config['allowed_referers'])) {
            $refererHost = parse_url($referer, PHP_URL_HOST);
            $allowed = false;
            
            foreach ($this->config['allowed_referers'] as $r) {
                if (strpos($refererHost, $r) !== false) {
                    $allowed = true;
                    break;
                }
            }
            
            if (!$allowed) {
                $this->score -= 20;
                $this->checks[] = 'invalid_referer';
            }
        }
    }
    
    /**
     * Verifica headers suspeitos
     */
    private function checkHeaders() {
        // Sem Accept-Language
        if (empty($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $this->score -= 10;
            $this->checks[] = 'no_language';
        }
        
        // Headers de proxy
        $proxyHeaders = [
            'HTTP_VIA',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'HTTP_X_PROXY_CONNECTION',
            'HTTP_PROXY_CONNECTION'
        ];
        
        foreach ($proxyHeaders as $h) {
            if (!empty($_SERVER[$h])) {
                $this->score -= 5;
                $this->checks[] = 'proxy_header';
                break;
            }
        }
    }
    
    /**
     * Verifica país
     */
    private function checkCountry($ip) {
        if (empty($this->config['blocked_countries'])) return;
        
        $country = $this->getCountry($ip);
        
        if (in_array($country, $this->config['blocked_countries'])) {
            $this->score -= 50;
            $this->checks[] = 'blocked_country:' . $country;
        }
    }
    
    /**
     * Adiciona headers de segurança
     */
    private function addSecurityHeaders() {
        // Anti-iframe
        if ($this->config['block_iframe']) {
            header('X-Frame-Options: DENY');
            header("Content-Security-Policy: frame-ancestors 'none'");
        }
        
        // Outros headers de segurança
        header('X-Content-Type-Options: nosniff');
        header('X-XSS-Protection: 1; mode=block');
        header('Referrer-Policy: strict-origin-when-cross-origin');
    }
    
    /**
     * Injeta JavaScript de proteção
     */
    private function injectProtection() {
        ob_start(function($html) {
            $js = $this->getProtectionJS();
            
            // Injeta antes do </head> ou no início do body
            if (strpos($html, '</head>') !== false) {
                $html = str_replace('</head>', "<script>$js</script></head>", $html);
            } elseif (strpos($html, '<body') !== false) {
                $html = preg_replace('/(<body[^>]*>)/i', "$1<script>$js</script>", $html);
            }
            
            return $html;
        });
    }
    
    /**
     * JavaScript de proteção client-side
     */
    private function getProtectionJS() {
        $blockDevtools = $this->config['block_devtools'] ? 'true' : 'false';
        $blockIframe = $this->config['block_iframe'] ? 'true' : 'false';
        $redirectUrl = $this->config['block_redirect'];
        
        return <<<JS
(function(){
    'use strict';
    
    var redirectUrl = '$redirectUrl';
    var blockDevtools = $blockDevtools;
    var blockIframe = $blockIframe;
    
    // Anti-iframe
    if (blockIframe && window.top !== window.self) {
        window.top.location = redirectUrl;
        return;
    }
    
    // Anti-DevTools
    if (blockDevtools) {
        // Detecta DevTools aberto
        var devtools = {open: false};
        var threshold = 160;
        
        setInterval(function() {
            if (window.outerWidth - window.innerWidth > threshold || 
                window.outerHeight - window.innerHeight > threshold) {
                if (!devtools.open) {
                    devtools.open = true;
                    window.location = redirectUrl;
                }
            }
        }, 500);
        
        // Bloqueia F12
        document.addEventListener('keydown', function(e) {
            if (e.keyCode === 123 || // F12
                (e.ctrlKey && e.shiftKey && (e.keyCode === 73 || e.keyCode === 74)) || // Ctrl+Shift+I/J
                (e.ctrlKey && e.keyCode === 85)) { // Ctrl+U
                e.preventDefault();
                return false;
            }
        });
        
        // Bloqueia clique direito
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
            return false;
        });
        
        // Detecta console aberto (debugger)
        (function() {
            var startTime = new Date();
            debugger;
            var endTime = new Date();
            if (endTime - startTime > 100) {
                window.location = redirectUrl;
            }
        })();
    }
    
    // Anti-copy
    document.addEventListener('copy', function(e) {
        e.preventDefault();
    });
    
    document.addEventListener('cut', function(e) {
        e.preventDefault();
    });
    
    // Desabilita seleção
    document.body.style.userSelect = 'none';
    document.body.style.webkitUserSelect = 'none';
    
    // Anti print-screen (parcial)
    document.addEventListener('keyup', function(e) {
        if (e.keyCode === 44) { // PrintScreen
            navigator.clipboard.writeText('');
        }
    });
    
    // Verifica se está em ambiente automatizado
    if (navigator.webdriver || 
        window.callPhantom || 
        window._phantom ||
        window.__nightmare ||
        navigator.plugins.length === 0) {
        window.location = redirectUrl;
    }
    
    // Fingerprint básico
    var fp = btoa(JSON.stringify({
        s: screen.width + 'x' + screen.height,
        d: screen.colorDepth,
        t: Intl.DateTimeFormat().resolvedOptions().timeZone,
        l: navigator.language,
        p: navigator.platform,
        c: navigator.hardwareConcurrency || 0,
        m: navigator.deviceMemory || 0
    }));
    
    // Envia fingerprint (opcional)
    // fetch('/api/fp.php', {method: 'POST', body: fp});
    
    console.log('%c⚠️ ÁREA PROTEGIDA', 'color: red; font-size: 30px; font-weight: bold;');
    console.log('%cEsta página está protegida contra clonagem.', 'color: red; font-size: 14px;');
    
})();
JS;
    }
    
    /**
     * Bloqueia acesso
     */
    private function block() {
        // Limpa output
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        header('HTTP/1.1 403 Forbidden');
        header('Location: ' . $this->config['block_redirect']);
        exit;
    }
    
    /**
     * Obtém IP real
     */
    private function getIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $h) {
            if (!empty($_SERVER[$h])) {
                $ip = explode(',', $_SERVER[$h])[0];
                return trim($ip);
            }
        }
        
        return '0.0.0.0';
    }
    
    /**
     * Obtém país do IP
     */
    private function getCountry($ip) {
        $cache = sys_get_temp_dir() . '/geo_' . md5($ip);
        
        if (file_exists($cache) && time() - filemtime($cache) < 86400) {
            return file_get_contents($cache);
        }
        
        $json = @file_get_contents("http://ip-api.com/json/{$ip}?fields=countryCode");
        $data = json_decode($json, true);
        $country = $data['countryCode'] ?? 'XX';
        
        @file_put_contents($cache, $country);
        
        return $country;
    }
    
    /**
     * Log de acessos
     */
    private function log($ip, $status, $checks) {
        if (!$this->config['enable_logs']) return;
        
        $dir = dirname($this->config['log_file']);
        if (!is_dir($dir)) @mkdir($dir, 0755, true);
        
        $log = date('Y-m-d H:i:s') . " | {$ip} | {$status} | " . 
               "Score: {$this->score} | " .
               implode(',', $checks) . " | " .
               ($_SERVER['HTTP_USER_AGENT'] ?? '-') . "\n";
        
        @file_put_contents($this->config['log_file'], $log, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Gera token para passar do Cloaker para o Checkout
     */
    public static function generateToken($secret = 'RIFA_PROTECT_2025') {
        return md5($secret . date('Y-m-d'));
    }
}

// ===========================================
// MODO DE USO
// ===========================================

/*
// No seu checkout/página real, adicione no TOPO:

require_once 'AntiClone.php';

$protection = new AntiClone([
    'secret_token' => 'RIFA_PROTECT_2025',  // Mesmo token do Cloaker
    'block_redirect' => 'https://google.com/',
    'allow_direct' => false,
    'block_iframe' => true,
    'block_devtools' => true,
    'block_bots' => true,
    'allowed_referers' => ['oferta01.com', 'rifamoto.com'],  // Seus domínios
]);

$protection->protect();

// Resto do seu código...
*/
